function S = tvaxrinfer(tvadata,tvamodel,theta,theta_fix)
% TVAXRINFER
%
%  Synopsis
%  ========
%
%  S = tvaxrinfer(tvadata,tvamodel,theta,theta_fix)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen
%     October 2010
%  
%  [1] Dyrholm, M., et al. (2011) "Single Trial Inference on Visual
%  Attention", AIP Conference Proceedings, Volume 1371, pp. 37-43
%  
%  Purpose
%  =======
%
%  For PR and WR compute distractor encoding probabilities
%  for individual trials, given observed values. 
%  See also TVAXRDDIST.
%
%  Output
%  ======
%
%  S{n}.Pi - The probability that element "i" is encoded
%  (Reported targets are set to 1 and non-reported targets 
%  are set to 0). 
%
%  S{n}.Ed - The expected number of distractors encoded. 
%
%  S{n}.Pd - A vector with the number of distractors 
%  encoded pdf.
%
%  S{n}.Es - Inferred VSTM load.

if nargin<4
  theta_fix = [];
end

if ~isempty(theta_fix)
  theta = tvathetacombine(theta,theta_fix);
end

N = length(tvadata);
[dum,logistidx,expidx] = tvahuman(theta,tvamodel);
S = {};
for n=1:N
  tvatrial = tvadata{n};
  % compute the VSTM probability and derivatives
  [v,taumu] = tvaflexchain(theta,tvamodel,tvatrial,logistidx,expidx,2);
  switch tvatrial.task
   case 'WR'
    [Pd,Ed,Pi] = tvaxrddist(taumu,v,tvamodel.K,tvatrial.response,1:length(v),tvamodel.s0); % =  [Pd,Ed,Pi] 
    Q = zeros(tvatrial.places,1);
    Q(tvatrial.display(tvatrial.response))=1;
    Pi = Q;
   case 'PR'
    [Pd,Ed,Pi] = tvaxrddist(taumu,v,tvamodel.K,tvatrial.response,tvatrial.targets,tvamodel.s0); % =  [Pd,Ed,Pi]
    Q = zeros(tvatrial.places,1);
    Q(tvatrial.display(tvatrial.response))=1;
    Q(tvatrial.display(tvatrial.distractors))=Pi;
    Pi = Q;
  end
  S{n}.Pi = Pi;
  S{n}.Ed = Ed;
  S{n}.Pd = Pd;
  S{n}.Es = sum(Pi);
end
