function [theta,tvamodel,theta_fix] = tvasculpt(theta,tvamodel,theta_fix,op,varargin)
% TVASCULPT
%
%  Synopsis
%  ========
%
%  [theta,tvamodel,theta_fix] = tvasculpt(theta,tvamodel,theta_fix,op)
%  [theta,tvamodel,theta_fix] = tvasculpt(theta,tvamodel,theta_fix,op,par1,par2,...)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - May 2011
%
%  Purpose
%  ======= 
%
%  Re-configure an existing tvamodel and theta pair.
%  
%  Inputs
%  ======
%
%  op - String indicating the operation to perform according
%  to the following table.
%
%   op string | Description
%   ----------+--------------------------------------------
%     'C2s'   | Go from model with C to model with s. 
%             | par1 is TVADATA, par2 is the 
%             | number of s values to use. par can be sinfl.
%     'wrep'  | Repeat w values. par1 - Repfactor.
%     'a2w'   | Go from scalar alpha to indiv. alpha.
%
%  (Note that theta_fix must be empty for some of these operations
%  to be successful due to the use of TVASETVAL)
%   
%  Outputs
%  =======
%
%  The outputs represent the re-configured model and parameters.

facstr = tvamodel.facstr;
gota = ismember('a',facstr);

switch op
 case 'C2s'
  if ismember('C',facstr)
    some = varargin{2};
    if isscalar(some)
      tmpall = size(tvamodel.infl,1);
      numloc = tvanumlocations(varargin{1});
      tvamodel.sinfl = tvalink(numloc,some);
    else
      tvamodel.sinfl = some;
      some = size(some,2);
    end
    if tvaisalphaexpanded(tvamodel)
      tvamodel.sinfl = cat(1,tvamodel.sinfl,tvamodel.sinfl);
    end
    [alpha,w,C] = tvadeal(tvamodel,theta);
    theta = tvasetval(theta,tvamodel,theta_fix,'C',repmat(C,[some,1]));
    tvamodel.facstr(find(tvamodel.facstr=='C')) = 's';
  else
    error('Cannot op=C2s when ''C'' not in facstr.');
  end
 case 'a2w'
  if ismember('a',facstr)
    [alpha,w] = tvadeal(tvamodel,tvahuman(tvathetacombine(theta,theta_fix),tvamodel));
    [theta,theta_fix] = tvasetval(theta,tvamodel,theta_fix,'a',[]);	
    facstr=strrep(facstr,'a','A');
    tvamodel.facstr = facstr;
    w = [1;w];
    w = tvafixer('+',[w;alpha*w]);
    [theta,theta_fix] = tvasetval(theta,tvamodel,theta_fix,'w',w(2:end));
    tvamodel.infl = [tvamodel.infl,zeros(size(tvamodel.infl));zeros(size(tvamodel.infl)),tvamodel.infl];
    if isfield(tvamodel,'sinfl')
      tvamodel.sinfl = cat(1,tvamodel.sinfl,tvamodel.sinfl);
    end
    %tvamodel.alphaexpanded = 1;
  else
    error('Cannot op=a2w when ''a'' not in facstr.');
  end
 case 'wrep'
  if ismember('w',facstr)
    [alpha,w] = tvadeal(tvamodel,theta,theta_fix);
    w = [0;w(:)];
    repfactor = varargin{1};
    w = repmat(w',[repfactor,1]); w=w(:);
    w = w(2:end);
    theta = tvasetval(theta,tvamodel,theta_fix,'w',w);
    [all,some] = size(tvamodel.infl);
    tvamodel.infl = kron(eye(repfactor),tvamodel.infl);
    tvamodel.infl = tvamodel.infl(1:repfactor:end,:);
  else
    error('Cannot op=wrep when ''w'' not in facstr.');
  end
 otherwise
  error('Unknown op string: %s',op);
end
