function [PP,gv,Hvv,gt,Hvt,Htt] = tvappar(tau,v,K,R,T,s0)
% TVAPPAR
%
%  Synopsis
%  ========
%
%  [PP,gv,Hvv,gt,Hvt,Htt] = tvappar(tau,v,K,R,T,s0)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen
%     February 2010
%  
%  Purpose
%  =======
%
%  This routine provides a safe wrapper to the TVACOREPPAR
%  routine which calls the libTVA kernel algorithm for
%  computing the partial report probability for the report R.
%  
%  Inputs
%  ======
%
%  tau - Expected effective stimulus duration. 
%  
%  v - Vector of v values.
%
%  K - Scalar, or vector indicating the VSTM capacity CDF.
%  If K is a scalar then the integer mix model is used, 
%  otherwise K must be the VSTM capacity CDF over 
%  [0,length(v)-1].
%
%  R - Vector, report, each element in [1,length(v)].
%
%  T - Vector, tagets, each element in [1,length(v)].
%
%  s0 - Scalar, optional, assumed deviation on t0. Specifying this
%  option s0>0.0 will enable the ex-gaussian model. If this option
%  is not specified, or set to 0.0 then exponential processing
%  will be used.
%
%  Outputs
%  =======
%  
%  PP - Report probability.
%
%  g* - Gradient arrays. gv is dPP/dv, gt is dPP/dt0.
%
%  H** - Hessian matrices. Hvv is ddPP/dvdv. Hvt is the 
%  vector ddPP/dvdt0, Htt is the scalar ddPP/dt0dt0.

% check inputs
if isscalar(K)
  K = cumsum(i2pdf(K,length(v)));
end
if length(K)<length(v)
  error('K must be either a scalar or have length greater than or equal to v');
end
if ~isempty(find(diff(sort(R))==0))
  error('R must not contain the same element multiple times.');
end
if ~isempty(find(diff(sort(T))==0))
  error('T must not contain the same element multiple times');
end
% compute
if nargin<5
  s0 = 0.0;
end
D = setdiff(1:length(v),T);
if isempty(D)
  warning('No distractors.');
end
[PP,gv,Hvv,gt,Hvt,Htt] = tvacoreppar(double(tau),double(v),double(K),int32(R-1),int32(T-1),int32(D-1),double(s0));
% fill lower triangle of Hvv
Hvv = triu(Hvv) + triu(Hvv,1)';


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
% AUXILIARY
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function histo = i2pdf(mu,N)
% I2PDF
%
%  Synopsis
%  ========
%
%  histo = i2pdf(mu,N)
%
%  -- Author: Mads Dyrholm, mistermads@gmail.com --
histo = zeros(1,N);
histo(ceil(mu)+1) = mu-floor(mu);
histo(floor(mu)+1) = max(ceil(mu),floor(mu+1))-mu;
