function [PE,gv,Hvv,gt,Hvt,Htt] = tvapenc(tau,v,K,x,s0)
% TVAPENC
%
%  Synopsis
%  ========
%
%  [PE,gv,Hvv,gt,Hvt,Htt] = tvapenc(tau,v,K,x)
%  [PE,gv,Hvv,gt,Hvt,Htt] = tvapenc(tau,v,K,x,s0)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen
%     2009 - January 2010
%  
%  Purpose
%  =======
%
%  This routine provides a safe wrapper to the TVACOREPENC
%  routine which calls the libTVA kernel algorithm for
%  computing the probability that an item is encoded into
%  VSTM.
%  
%  Inputs
%  ======
%
%  tau - Expected effective stimulus duration, t-E[t0].
%  
%  v - Vector of v values.
%
%  K - Scalar, or vector indicating the VSTM capacity CDF.
%  If K is a scalar then the integer mix model is used, 
%  otherwise K must be the VSTM capacity CDF over 
%  [0,length(v)-1].
%
%  x - Probed item scalar in [1,length(v)].
%
%  s0 - Scalar, optional, assumed deviation on t0. Specifying this
%  option s0>0.0 will enable the ex-gaussian model. If this option
%  is not specified, or set to 0.0 then exponential processing
%  will be used (i.e. t0=E[t0]).
%
%  Outputs
%  =======
%  
%  PE - Encoding probability.
%
%  g* - Gradient arrays. gv is dPE/dv, gt is dPE/dE[t0].
%
%  H** - Hessian matrices. Hvv is ddPE/dvdv. Hvt is the 
%  vector ddPE/dvdE[t0], Htt is the scalar ddPE/dE[t0]dE[t0].

% check inputs
if isscalar(K)
  K = cumsum(i2pdf(K,length(v)));
end
if length(K)<length(v)
  error('K must be either a scalar or have length greater than or equal to v');
end
% compute
if nargin<5
  s0 = 0.0;
end
[PE,gv,Hvv,gt,Hvt,Htt] = tvacorepenc(double(tau),double(v),double(K),int32(x-1),double(s0));
% fill lower triangle of Hvv
Hvv = triu(Hvv) + triu(Hvv,1)';


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
% AUXILIARY
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function histo = i2pdf(mu,N)
% I2PDF
%
%  Synopsis
%  ========
%
%  histo = i2pdf(mu,N)
%
histo = zeros(1,N);
histo(ceil(mu)+1) = mu-floor(mu);
histo(floor(mu)+1) = max(ceil(mu),floor(mu+1))-mu;
