function tvamodel = tvam(tvamodel,str,varargin)
% TVAM
%
%  Synopsis
%  ========
%
%         m = tvam([]      ,str,par1,par2,...)
%  tvamodel = tvam(tvamodel,str,par1,par2,...)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - May 2010
%  
%  Purpose
%  =======
%
%  Function for setting the VSTM capacity histogram and
%  lapse probability.
%
%  Inputs
%  ======
%
%  str - A string indicating which pdf to use and
%  the meaning of par1 etc. Valid strings are:
%  
%    'TRAD' - (par1)=(mu)
%  
%    'BINOSTACK' - (par1,par2,par2)=(Kalways,Kmaybe,Pmaybe). The
%    mean of this is Kalways+Kmaybe*Pmaybe.
%
%    'm' - (par1)=(m)
%
%    'lapse' - (par1)=(Plapse)  , requires tvamodel given on input.
%
%  Outputs
%  =======
%
%  If tvamodel is [] then the pdf is returned. Otherwise the cdf is
%  returned at tvamodel.K

if isempty(tvamodel)
  N = 100;
else
  N = length(tvamodel.K);
end

switch upper(str)
 case 'TRAD'
  mu = varargin{1};
  m = i2pdf(mu,N);
 case 'BINOSTACK'
  kmin = varargin{1};
  kmay = varargin{2};
  P = varargin{3};
  m = [zeros(1,kmin) binopdf(0:N-kmin-1,kmay,P)];
 case 'M'
  m = varargin{1};
  m = m/sum(m);
  m = cat(1,m(:),zeros(N-length(m),1))';
 case 'LAPSE'
  [muk,m] = tvamuk(tvamodel);
  Plapse = varargin{1};
  b = m(2:end);
  x = sum(b) / (1-Plapse);
  b = b / x;
  m = [Plapse,b];
 otherwise
  error('Unknown str given.');
end

if isempty(tvamodel)
  tvamodel = m;
else
  tvamodel.K = cumsum(m);
end
  
function histo = i2pdf(mu,N)
% I2PDF
%
%  Synopsis
%  ========
%
%  histo = i2pdf(mu,N)
%
%  -- Author: Mads Dyrholm --
%
%  Purpose
%  =======

histo = zeros(1,N);
histo(ceil(mu)+1) = mu-floor(mu);
histo(floor(mu)+1) = max(ceil(mu),floor(mu+1))-mu;
