function [tvadata,columns] = tvaloader(datafile,what,opts)
% TVALOADER
%
%  Synopsis
%  ========
% 
%  [tvadata,columns] = tvaloader(datafile)
%  [tvadata,columns] = tvaloader(datafile,what)
%  [tvadata,columns] = tvaloader(datafile,what,opts)
%  
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - December 2012
%  
%  Purpose
%  =======
%
%  Load data from file.
%
%  Inputs
%  ======
%
%  datafile - A string containing the data filename.
%
%  what - A string, optional, not case sensitive, 
%  possible values: 'STD','CDTAS'. Default is STD.
%
%  opts - Depends on 'what' as following table
%
%   what  | Description of opts
%  -------+------------------------------------------
%   'STD' | Vector with un-masked condition numbers.
%
%  Outputs
%  =======
%
%  tvadata - struct

fprintf('File: %s\n',datafile);

if nargin>1 & strcmp(what,'IRISCUE')
    STD_flag_ignoredistractors = 1;
    fprintf('(Iris: distractors will be ignored!)\n');
    what='STD';
else
    STD_flag_ignoredistractors = 0;
end

WHSTR='STD';
if nargin<2 || isempty(what)
    fprintf('Format: Autodetect standard TVA formats.\n');
    what = WHSTR;
end

if nargin<3
    opts = [];
end

switch upper(what)
  case 'KATPEST'
    fprintf('Format: Katharina PEST data.\n');
    qqq=textread(datafile,'%s');
    M = 8;
    t = str2double(qqq(1:M:end));
    hit = str2double(qqq(7:M:end));
    resp = qqq(5:M:end);
    N = length(t);
    for n=1:N
        tvadata{n}.display = 1;
        tvadata{n}.task = 'WR';
        tvadata{n}.targets = 1;
        tvadata{n}.distractors = [];
        tvadata{n}.t = t(n);
        if hit(n)==1
            tvadata{n}.response = 1;
        else
            tvadata{n}.response = [];
        end
        tvadata{n}.places = 1;
        tvadata{n}.more.condition = find(tvadata{n}.t==sort(unique(t)));
        tvadata{n}.more.errors_extra = (resp{n}~='-')*(1-hit(n));
        tvadata{n}.more.errors_intrusion = 0;
        tvadata{n}.more.report_len_unique = resp{n}~='-';
    end
  case 'CDTASNEW'
    qqq=textread(datafile,'%s');
    N = str2num(qqq{1});
    qqq = qqq(2:end-mod(length(qqq)-1,N));
    columns = length(qqq)/N;
    www=reshape(qqq,columns,length(qqq)/columns)';  
    tvadata = {};
    switch columns
      case 5  % 5 columns TAS CD
        fprintf('Format: TAS New CD data format.\n');
        for n=1:N 
            [qq,qw,qe,qr,qt]=deal(www{n,:});
            tvadata{n}.response = str2num(qr);
            if tvadata{n}.response==5
                tvadata{n}.response=2;
            end
            tvadata{n}.display = find(qw~='0');
            tvadata{n}.probe = find(tvadata{n}.display==str2num(qe));
            tvadata{n}.change = str2num(qt);
            tvadata{n}.targets = 1:length(tvadata{n}.display);
            tvadata{n}.distractors = [];
            tvadata{n}.task = 'CD';
            tvadata{n}.t = str2num_safe(qq);
            tvadata{n}.places = length(qw);
            tvadata{n}.condition = 10*round(tvadata{n}.t/10) + tvadata{n}.change;
        end    
      otherwise
        fprintf('Format: Could not detect data format.\n');
    end
  case 'CDTAS'
    %fprintf('Format: Non-standard TAS Change Detection.\n');
    qqq=textread(datafile,'%s');
    N = str2num(qqq{1});
    qqq = qqq(2:end-mod(length(qqq)-1,N));
    columns = length(qqq)/N;
    %fprintf('Found %i columns in the file.\n',columns);
    www=reshape(qqq,columns,length(qqq)/columns)';  
    tvadata = {};
    switch columns
      case 5  % 5 columns TAS CD
        fprintf('Format: Detected non-standard 5 column TAS Change Detection data format.\n');
        for n=1:N 
            [qq,qw,qe,qr,qt]=deal(www{n,:});
            tvadata{n}.response = str2num(qt);
            if tvadata{n}.response==5
                tvadata{n}.response=2;
            end
            tvadata{n}.display = find(qe~='0');
            tvadata{n}.probe = find(qr(tvadata{n}.display)~='0');
            tvadata{n}.change = isempty(find(qe==qr));
            tvadata{n}.targets = 1:length(tvadata{n}.display);
            tvadata{n}.distractors = [];
            tvadata{n}.task = 'CD';
            tvadata{n}.t = str2num_safe(qw);
            tvadata{n}.places = length(qe);
        end    
      otherwise
        fprintf('Format: Could not detect data format.\n');
    end
  case WHSTR;
    qqq=textread(datafile,'%s');
    N = str2double(qqq{1});
    qqq = qqq(2:end-mod(length(qqq)-1,N));
    columns = length(qqq)/N;
    %fprintf(sprintf('Found %i columns in the file.\n',columns));
    www=reshape(qqq,columns,length(qqq)/columns)';
    tvadata = {};  
    switch columns
      case 12  % 12 columns Celine format
        fprintf('Format: Non-standard CG Change Detection with 12 columns.\n');
        for n=1:N %100 %N
            [qq,qw,qe,qr,qt,qy,qu,qi,qo,qp,qa,qs]=deal(www{n,:});
            tvadata{n}.condition = str2num(qq);
            tvadata{n}.change = str2num(qo) - 1;
            tvadata{n}.response = str2num(qa) - 1;
            bd = (qt~='0') | (qy~='0');
            bdidx = 1:length(bd);
            tvadata{n}.display = find(bd);
            xx = find(qi~='0'); % probe
            tvadata{n}.probe = find(tvadata{n}.display==xx);	    
            qt = qt(bd);
            tvadata{n}.targets = bdidx(find(qt~='0'));
            qy = qy(bd);    
            tvadata{n}.distractors = bdidx(find(qy~='0'));
            tvadata{n}.task = 'CD';
            tvadata{n}.t = str2num_safe(qw);
            tvadata{n}.places = length(bd);
        end
        
        %    idxnot3 = find(rr>=0 & rr<=2);
        %    [xx,cc,rr,setS,tt] = deal(xx(idxnot3),cc(idxnot3),rr(idxnot3),setS(idxnot3),tt(idxnot3));
        
      case 5   % 5 columns TVA format
        fprintf('Format: Detected standard 5 columns TVA format.\n');
        for n=1:N
            
            [qq,qw,qe,qr,qt]=deal(www{n,:});
            
            if strcmp(qw,'111111') % this is to detect TAS' very skumle dataformat.
                tvadata{n}.response = str2num(qr);
                if tvadata{n}.response==5
                    tvadata{n}.response=2;
                end
                tvadata{n}.display = find(qw~='0');
                tvadata{n}.probe = find(tvadata{n}.display==str2num(qe));
                tvadata{n}.change = str2num(qt);
                tvadata{n}.targets = 1:length(tvadata{n}.display);
                tvadata{n}.distractors = [];
                tvadata{n}.task = 'CD';
                tvadata{n}.t = str2num_safe(qq);
                tvadata{n}.places = length(qw);
                tvadata{n}.condition = 10*round(tvadata{n}.t/10) + tvadata{n}.change;
                continue % forget about it
            end
            
            isCDtrial = ~isempty(intersect( find(qe~='0'),find(qr~='0')));
            
            if isCDtrial  % CD TAS auto
                tvadata{n}.response = str2num(qt);
                if tvadata{n}.response==5
                    tvadata{n}.response=2;
                end
                tvadata{n}.display = find(qe~='0');
                tvadata{n}.probe = find(qr(tvadata{n}.display)~='0');
                tvadata{n}.change = isempty(find(qe==qr));
                tvadata{n}.targets = 1:length(tvadata{n}.display);
                tvadata{n}.distractors = [];
                tvadata{n}.task = 'CD';
                tvadata{n}.t = str2num_safe(qw);
                tvadata{n}.places = length(qe);
                
            else % WR or PR auto
                
                if ((length(qr)==1) && (qr=='0')) % fix for Annemaries single-0 no-target coding
                    qr=repmat('0',size(qe));
                end
                if STD_flag_ignoredistractors==1 % Iris cue
                    qr=repmat('0',size(qe));
                end
                
                tvadata{n}.distractors = find(qr~='0');
                tvadata{n}.targets = find(qe~='0');
                
                qt = setdiff(qt,union('0','-'));
                errors_intrusion = length(intersect(qr,qt));
                errors_extra = length(setdiff(qt,union(qr,qe)));
                report_len_orig = length(qt);
                report_len_unique = length(unique(qt));
                
                bd = (qr~='0') | (qe~='0');
                bdidx = 1:length(bd);
                tvadata{n}.display = find(bd);
                qr = qr(bd);
                qe = qe(bd);    
                tvadata{n}.targets = bdidx(find(qe~='0'));
                tvadata{n}.distractors = bdidx(find(qr~='0'));
                
                if ~isempty(intersect( tvadata{n}.distractors, tvadata{n}.targets))
                    error(sprintf('Trial #%i: An element was both target and distractor.',n));
                end
                
                if isempty(tvadata{n}.distractors)
                    tvadata{n}.task = 'WR';
                else
                    tvadata{n}.task = 'PR';
                end
                
                tvadata{n}.t = str2num_safe(qw);
                
                % figure out target R relative to display
                qeshrink = qe(tvadata{n}.targets);    
                R = qt;
                tmp = [];
                for r=1:length(R)
                    tmp = [tmp find(qeshrink==R(r))];
                end
                tmp = sort(tmp);
                repidx = find(diff(tmp)==0);
                if ~isempty(repidx)
                    warning(sprintf('Trial #%i: Subject reported the same element more than once. Redundancy removed.',n));
                    tmp = unique(tmp);
                end
                tvadata{n}.response = tvadata{n}.targets(tmp);
                tvadata{n}.places = length(bd);
                tvadata{n}.more.condition = str2double(qq);
                tvadata{n}.more.errors_extra = errors_extra;
                tvadata{n}.more.errors_intrusion = errors_intrusion;
                tvadata{n}.more.report_len_orig = report_len_orig;
                tvadata{n}.more.report_len_unique = report_len_unique;
            end
        end
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      case 6   % 6 columns TVA format with unmasked flag
        fprintf('Format: Detected Annemaries 6 columns TVA format.\n');
        for n=1:N
            [qq,qw,qe,qr,qt,qy]=deal(www{n,:});
            
            % fix for Annemaries single-0 no-target coding
            if ((length(qr)==1) & (qr=='0')), 
                qr=repmat('0',size(qe)); 
            end
            if STD_flag_ignoredistractors==1 % Iris cue
                qr=repmat('0',size(qe));
            end
            
            if strcmp(qy,'1')
                tvadata{n}.unmask = 1;
            end
            
            tvadata{n}.distractors = find(qr~='0');
            tvadata{n}.targets = find(qe~='0');
            
            qt = setdiff(qt,union('0','-'));
            errors_intrusion = length(intersect(qr,qt));
            errors_extra = length(setdiff(qt,union(qr,qe)));
            report_len_orig = length(qt);
            report_len_unique = length(unique(qt));
            
            bd = (qr~='0') | (qe~='0');
            bdidx = 1:length(bd);
            tvadata{n}.display = find(bd);
            qr = qr(bd);
            qe = qe(bd);    
            tvadata{n}.targets = bdidx(find(qe~='0'));
            tvadata{n}.distractors = bdidx(find(qr~='0'));
            
            if ~isempty(intersect( tvadata{n}.distractors, tvadata{n}.targets)), 
                error(sprintf('Trial #%i: An element was both target and distractor.',n)); 
            end
            
            if isempty(tvadata{n}.distractors)
                tvadata{n}.task = 'WR';
            else
                tvadata{n}.task = 'PR';
            end
            
            tvadata{n}.t = str2num_safe(qw);
            
            % figure out target R relative to display
            qeshrink = qe(tvadata{n}.targets);    
            R = qt;
            tmp = [];
            for r=1:length(R),  tmp = [tmp find(qeshrink==R(r))]; end
            tmp = sort(tmp);
            repidx = find(diff(tmp)==0);
            if ~isempty(repidx)
                warning(sprintf('Trial #%i: Subject reported the same element more than once. Redundancy removed.',n));
                tmp = unique(tmp);
            end
            tvadata{n}.response = tvadata{n}.targets(tmp);
            tvadata{n}.places = length(bd);
            tvadata{n}.more.condition = str2double(qq);
            tvadata{n}.more.errors_extra = errors_extra;
            tvadata{n}.more.errors_intrusion = errors_intrusion;
            tvadata{n}.more.report_len_orig = report_len_orig;
            tvadata{n}.more.report_len_unique = report_len_unique;
        end
    end 
    % optional un-masked conditions
    if ~isempty(opts), tvadata = tvapatcher(tvadata,{opts,'unmask',1}); end
  otherwise
    error('Error: Unknown data format specified, see help banner for possible values.\n');
end

% AUX
function n = str2num_safe(s)
s(find(s==',')) = '.';
n = str2double(s);
    
    
