function opts = tvagetopts(datafiles)
% TVAGETOPTS
%
%  Synopsis
%  ========
% 
%  opts = tvagetopts('datafile 1', 'datafile 2', ...)
%  
%  -- Author: Rasmus Berg Palm --
%     Center for Visual Cognition, University of Copenhagen.
%     May 2011
%  
%  Purpose
%  =======
%
%  Returns intersect of possible options for datafiles to be used in TVAFIT. If only one datafile is suplied it returns the possible options for that datafile.
%
%  Inputs
%  ======
%
%  datafile(s) - string filename(s)
%
%  Outputs
%  =======
%
%  opts - struct with possible options for TVAFIT. The first element of the options are the default values.
%
%  Example
%  =======
%  opts = tvagetopts('betula.dat', 'betula_bad.dat')
% 
%  opts = 
% 
%     mode: 'attentional weights'
%        L: 1
%        M: {'FREE'  'TRAD'  'BINOSTACK'}
%        C: {'EXP'  'EX-GAUSSIAN'}
%        P: {[0]  [1]}
%        S: [0 1]
%        A: 1
    assert(iscellstr(datafiles), 'Inputs must be filenames to datafiles');

    %{
    
>> [N,e] = tvanumlocations(tvadata);
>> B = tvalink(N);
>> facstr = tvasniff(tvadata);
    
e skal være nul. Det betyder at alle trials i datasættet har N
positions (de er ikke nødvendligvis i brug i alle trials).

Setup gui
=======

1. L, 'Number of ###'

Hvis facstr indeholder et 'w' så skal ### være 'attentional weights',
ellers skal ### være 'v values'.
L er så 'antallet af ###'. B er de mulige værdier for L. Der skal
således findes intersect
mellem B for alle datasæt.

2. M, 'K-Model'

Man kan vælge 'TRAD','FREE',BINOSTACK'.
Men, hvis der altid kun vises 1 element på skærmen så skal man vælge
FREE. Det kan checkes ved at se om facstr indeholder et 'v'. Hvis den
gør, så skal brugeren ikke presenteres med valget (men brug 'FREE' til
det underliggende kald af TVAFIT).

3. C, 'Curve'

'Exponential' eller 'Ex-Gaussian'

4. P, 'Estimate proportion of lapses?'

'Yes, estimate proportion of lapses' eller 'No, do not estimate
proportion of lapses'.
    
S. 'Split C'

Skal være tilgængelig hvis facstr indeholder et 'C'. Den skal have
valgmulighederne i B.
    
A, 'Alpha'

Skal være en mulighed hvis facstr indeholder et 'a'.
'One alpha per ###', eller 'One alpha covering all ###'.
%}
    
    [e,B,facstr] = deal(cell(nargin,1));
    for i = 1:nargin
        tvadata = tvaloader(datafiles{i});
        
        [N, e{i}] = tvanumlocations(tvadata);
        B{i} = tvalink(N);
        facstr{i} = tvasniff(tvadata);

    end
    a = find(cell2mat(e));
    assert(isempty(a), ['datafile(s) ' num2str(a) ' did not have the same number of positions in all rows']);
    
    % Find intersects
    if(nargin > 1)
        common_facstr = intersectall(facstr);
        common_B = intersectall(B);
    else
        common_facstr = facstr{1};
        common_B = B{1};
    end
    
    assert(~isempty(common_facstr), 'The datafiles can not be fitted with the same options: Their facstr intersect is empty. See TVASNIFF.');
    
    is_w = ~isempty(strfind(common_facstr,'w'));
    is_v = ~isempty(strfind(common_facstr,'v'));
    assert(is_v == 1 || is_w == 1, 'The datafiles can not be fitted with the same options: They have no common fitting mode (attentional weights or v values). See TVASNIFF.');
    modes = {'attentional weights', 'v values'};
    mode = modes{[is_w is_v]};
    
    assert(~isempty(common_B), ['The datafiles can not be fitted with the same options: they have no common number of ' mode '. See TVALINK.']);
    
    opts.mode = mode;
    
    opts.L = common_B;
    
    if(is_v == 1)
        opts.M = {'FREE'};
    else
        opts.M = {'FREE','TRAD','BINOSTACK'};
    end
    
    opts.C = {'EXP','EX-GAUSSIAN'};
    
    opts.P = {0, 1};
    
    opts.S = 0;
    if(~isempty(strfind(common_facstr,'C')))
        opts.S = [0 common_B];
    end
    
    opts.A = 1;
    if(~isempty(strfind(common_facstr,'a')))
        opts.A = {1, 2};
    end
    
    function res=intersectall(c)
        if length(c) == 2 
            res = intersect(c{1}, c{2}); 
            return 
        end
        res = intersect(c{1},intersectall(c(2:end)));
    end
    
end