function [v,theta_fix] = tvafixer(s,x,idx,tvamodel)
% TVAFIXER
%
%  Synopsis
%  ========
%
%  1.                 v = tvafixer(s,x)
%  2. [theta,theta_fix] = tvafixer(theta,theta_fix,idx,tvamodel)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - April 2010
%
%  Purpose
%  ======= 
%
%  Useful for fixing individual parameters:
%
%  1. Transform a value 'x' to so that it can be put in theta 
%  or theta_fix manually.
%  2. Fix a set of parameters in theta. The fixed parameters
%  are moved from theta to theta_fix.
%
%  Inputs
%  ======
%
%  s - A character, can be '+' or 'u'. Set it to chose
%  the transformation of x to v.
%
%  x - A number to be transformed. Can be a vector, then all
%  elements will be subject to the same transformation.
%
%  theta - Parameter vector.
%
%  theta_fix - Vector of fixed parameters.
%  
%  idx - Parameter indices to fix. The indices are relative
%  to the combination of theta and theta_fix. That is, if
%  theta_fix is [] then idx is relative to theta. Otherwise
%  idx is relative to theta_fix.
%
%  tvamodel - Model struct. (see also TVAINIT)
%
%  Outputs
%  =======
%
%  v - Transformed version of x. If s is '+' then the 
%  exponential function is used. If s is 'u' then the
%  logistic function is used.
%
%  theta - New theta, with fewer parameters than on input.
%
%  theta_fix - New theta_fix containing the fixed parameters.

if nargin==2
  % 1.
  x=x(:);
  switch s
   case '+'
    if x == 0
      v = -inf(size(x));
    else
      v = log(x);
    end
   case 'u'
    if x == 0 
      v = -inf(size(x));
    elseif x == 1
      v = inf(size(x));
    else
      v = -log(1./x-1);
    end
   otherwise
    error('s not recognized');
  end
elseif nargin==4
  % 2.
  theta = s;
  theta_fix = x;
  [theta,nanfix,notnanfix] = tvathetacombine(theta,theta_fix);
  theta_fix = nan(size(theta));
  fixidx = unique([notnanfix(:);idx(:)]);
  theta_fix(fixidx) = theta(fixidx);
  theta = theta(find(isnan(theta_fix)));
  v = theta;
else
  error('Wrong number of input arguments.');
end
