function [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L,M,C,P,S,A)
% TVAFIT
%
%  Synopsis
%  ========
%
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D)
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L)
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L,M)
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L,M,C)
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L,M,C,P)
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L,M,C,P,S)
%  [theta,tvamodel,tvadata,df,bic,theta_fix] = tvafit(D,L,M,C,P,S,A)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - October 2013
%  
%  Purpose
%  =======
%
%  TVA fitting made easy.
%  
%  Inputs
%  ======
%
%  (Note that any of the inputs can be set to [] for 
%  default value)
%
%  D - a FILENAME string or TVADATA struct. If a string is
%  given, then the file will be loaded using TVALOADER.
%
%  L - Scalar or matrix. Either the number of hazard rates,
%  the number of attentional weights, or the inflator matrix.
%  Default value is 1. See also TVALINK.
%
%  M - String indicating the K model. Possible values are
%  'FREE','BINOSTACK','TRAD'. Default value is 'FREE'.
%
%  C - String indicating the shape of the curve. Possible 
%  values are 'EXP' and 'EX-GAUSSIAN'. Default value is 
%  'EXP'.
%  
%  P - Scalar, flag whether lapse probability should be 
%  estimated. Set to 1 for yes. Default value is 0.
%  
%  S - Scalar, indicating split processing capacity. Set
%  equal to the number of s values to estimate. Default value
%  is 0 which means that C is estimated instead of s.
%
%  A - Scalar, flag whether alpha is a scalar or whether 
%  distractors get their own weight. Set to 1 for scalar
%  alpha, set different to enable individual distractor 
%  weights. Default value is 1.
%
%  Outputs
%  =======
%
%  df - Degrees of freedom used.
%
%  bic - Bayes Information Criterion, see also TVABIC.
%
%  tvamodel.M - Set to M.
%
%  tvamodel.Curve - Set to C.

if nargin<2 || isempty(L)
  L = 1;
end

if nargin<3 || isempty(M)
  M = 'FREE';
end

if nargin<4 || isempty(C)
  C = 'EXP';
end

if nargin<5 || isempty(P)
  P = 0;
end

if nargin<6 || isempty(S)
  S = 1;
end

if ~isscalar(S) % matrix S
  sinfl = S;
  S = size(S,2);
else
  sinfl = S;
end

if nargin<7 || isempty(A)
  A = 1;
end

if ischar(D)
  tvadata=tvaloader(D);
else
  tvadata=D;
end

locations = tvanumlocations(tvadata);
if length(L)==1
  L = tvalink(locations,L);
end
infl = L;

[theta,tvamodel] = tvainit(tvadata);

if isfield(tvamodel,'t0rem') && tvamodel.t0rem==1
    [~,~,~,~,~,idx_u0] = tvadeal(tvamodel,1:length(theta));
    [theta,theta_fix] = tvafixer(theta,[],idx_u0,tvamodel);
else
    theta_fix=[];
end

%if 0  % make gm adaptive ?
if isfield(tvamodel,'chdetgm')
  theta = [theta;tvafixer('+',tvamodel.chdetgm(:))];
  tvamodel.chdetgm = length(tvamodel.chdetgm); % CD1 or CD2 
  if tvamodel.chdetgm==1, tvamodel.chdetgm=[]; end % CD1
end
%end

[theta,tvamodel,change] = tvashave(theta,tvamodel,tvadata,theta_fix,[0.5 100 0.01]);
tvamodel.infl = infl;
gotalpha = ~isempty(strfind(tvamodel.facstr,'a'));
gotw = ~isempty(strfind(tvamodel.facstr,'w'));
if gotalpha
  alphatmp = theta(1);
  theta = theta(2:end);
else
  alphatmp = [];
end
expand = size(infl,2);
if gotw
  theta = [alphatmp;
	   tvafixer('+',ones(expand-1,1));
	   theta];
else
  theta = [alphatmp;
	   repmat(theta(1),[expand-1,1]);
	   theta];
end
if ~isempty(theta_fix)
    theta_fix = [nan(length(alphatmp));repmat(theta_fix(length(alphatmp)+1),[expand,1],1);theta_fix(length(alphatmp)+2:end)];
end
[theta,info] = tvaputt(theta,tvamodel,tvadata,theta_fix,1000);

for coasc = 1:1
  switch upper(M)
   case 'FREE'
    [theta,tvamodel,change] = tvashave(theta,tvamodel,tvadata,theta_fix,[0.1  1000 0.01]);
    [theta,tvamodel,change] = tvashave(theta,tvamodel,tvadata,theta_fix,[0.02 1000 0.01]);
    [dummuk,dumm,dumidx] = tvamuk(tvamodel);
    dfM = length(dumidx)-1;
   case 'BINOSTACK'
    [theta,tvamodel,Ka,Km,Pm] = tvasweepbinostack(theta,tvamodel,tvadata);
    dfM = 3;
   case 'TRAD'
    [theta,tvamodel] = tvasweeptrad(theta,tvamodel,tvadata);
    dfM = 1;
   otherwise
    error('Invalid value for M given.');
  end
  % lapse
  if P==1
    [tvamodel,theta] = tvalapse(tvamodel,theta,tvadata);
    dfP = 1;
  else
    dfP = 0;
  end
  if coasc==1
    % s?
    fl = 0;
    if ((S>1) && (gotw==1))
      fl=1;
      %keyboard
      [theta,tvamodel] = tvasculpt(theta,tvamodel,theta_fix,'C2s',tvadata,sinfl);
    end
    % vector alpha?
    if ((A~=1) && (gotalpha==1))
      fl=1;
      [theta,tvamodel] = tvasculpt(theta,tvamodel,theta_fix,'a2w');
    end
    if fl==1, [theta,info] = tvaputt(theta,tvamodel,tvadata); end
  end
  df = length(theta);
  % curve
  switch upper(C)
   case 'EX-GAUSSIAN'
    [theta,tvamodel] = tvaheater(theta,tvamodel,tvadata);
    dfC = 1;
   case 'EXP'
    dfC = 0;
   otherwise
    error('Invalid value for C given.');
  end
end

df = df + dfM + dfP + dfC;
if nargout>4
  nll = tvacost(theta,theta_fix,tvamodel,tvadata);
  bic = tvabic(tvadata,df,-nll);
end

tvamodel.M = M;
tvamodel.Curve = C;
