function tvaexport(filename,varargin) %par1,par2,par3,par4,par5)
% TVAEXPORT
%
%  Synopsis
%  ========
% 
%  tvaexport(filename,op,par1,par2,...,op,...)
%  
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - November 2010
%  
%  Purpose
%  =======
%
%  Export fits to text file suitable for importing into
%  statistical software such as R, SPSS, etc. 
%  The exported file comes with a header, followed by a 
%  line per fit.
%
%  Inputs
%  ======
%
%  filename - The file to create/overwrite. TVAEXPORT Will
%  prompt if the file already exists.
%
%  op - String indicating which option the following arguments
%  will pertain to. Options are divided in two groups:
%
%   Group 1. How to pass the data and fits to TVAEXPORT.
%
%    op     | Description
%   --------+----------------------------------------
%    'DIR'  | par1 is the dir argument and the export
%           | will be performed on the files that
%           | match the dir argument. Each file must
%           | have a tvadata, tvamodel, theta, 
%           | and optionally theta_fix variable.
%    'GIVE' | Give all the fits as five cell arrays of 
%           | identical dimension: par1 is a cell 
%           | array with par1{#} being TVADATA for
%           | fit no. #. Similarly, par2 holds TVAMODEL,
%           | par3 holds THETA, par4 holds the id
%           | strings, and par5 holds THETA_FIX.
%           | As an option, if par5 is empty then 
%           | theta_fix is assumed [].
%
%   Group 2. What to put in the exported report.
%
%    op     | Description
%   --------+----------------------------------------
%   'STL'   | Single-trial info will be exported.
%   'LPR'   | Fitted parameters will be exported.
%
%   The default for Group 2 is LPR.

if exist(filename,'file')
  warning(sprintf('The file ''%s'' already exists!',filename));
  answer = input('Are you sure that you want to continue and overwrite the existing file? [y/n]','s');
  if ((length(answer)~=1) || (answer~='y')), fprintf('User did not answer ''y'', exiting.\n'); return, end
  delete(filename); % delete file
end

SINGLETRIAL=1;
LPR=0;
outflag = LPR;

i=1;
while i<=length(varargin)
  switch upper(varargin{i})
   case 'DIR'
    par = varargin{i+1};
    parpath = fileparts(par);
    qq=dir(par);
    if length(qq)==0, fprintf('DIR option found no files to work with.\n'); return, end
    for subj=1:length(qq)
      fname = fullfile(parpath,qq(subj).name);
      fprintf('Loading file: %s\n',fname);
      theta_fix = [];
      load(fname);
      huske_theta_fix{subj} = theta_fix;
      huske_theta{subj} = theta;
      huske_tvadata{subj} = tvadata;
      huske_tvamodel{subj} = tvamodel;
      huske_id{subj} = qq(subj).name;
    end
    i = i+2;
   case 'GIVE'
    huske_tvadata   = varargin{i+1};
    huske_tvamodel  = varargin{i+2};
    huske_theta     = varargin{i+3};
    huske_id        = varargin{i+4};
    huske_theta_fix = varargin{i+5};
    i = i+6;
   case 'STL'
    outflag = SINGLETRIAL;
    i=i+1;
   case 'LPR'
    outflag = LPR;
    i=i+1;
   otherwise
    error('Invalid op argument.');
  end
end

fprintf('Exporting to %s\n',filename);
N = length(huske_tvadata);
for cl=1:N
  fprintf('.');
  tvadata = huske_tvadata{cl};
  tvamodel = huske_tvamodel{cl};
  theta = huske_theta{cl};
  tvaname = huske_id{cl};
  if isempty(huske_theta_fix), theta_fix=[]; else theta_fix = huske_theta_fix{cl}; end
  switch outflag
   case SINGLETRIAL
    tvaexportsingletrial(filename,tvaname,tvadata,tvamodel,theta,theta_fix);
   case LPR
    if cl==1
      tvalpr(filename,'',tvadata,tvamodel,theta,theta_fix);
    end
    tvalpr(filename,tvaname,tvadata,tvamodel,theta,theta_fix);
  end
end
fprintf('\n');

