function [humtheta,sigma,ll,pM,p,normg,H,g,U,rcondU] = tvadiag(theta,tvamodel,tvadata,theta_fix)
% TVADIAG
%
%  Synopsis
%  ========
%
%  [humtheta,sigma,ll,pM, ...
%   p,normg,H,g,U,rcondU] = tvadiag(theta,tvamodel,tvadata)
%  [humtheta,sigma,ll,pM, ...
%   p,normg,H,g,U,rcondU] = tvadiag(theta,tvamodel,tvadata,theta_fix)
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - February 2011
%
%  Purpose
%  ======= 
%
%  Compute human domain error bars and other diagnostics.
%
%  Inputs
%  ======
%
%  theta - Parameter vector. (see also TVAFLEXCHAIN)
%
%  tvamodel - Model struct. (see also TVAINIT)
%
%  tvadata - Dataset. (see also TVALOADER)
%
%  theta_fix - (Optional) Fixed parameters. Can be [] for 
%  no fixed parameter. (see also TVAFIXER)
% 
%  Outputs
%  =======
%
%  humtheta - Human readable theta combined with theta_fix. 
%  (see also TVAHUMAN)
%
%  sigma - Error bars for humtheta. Elements corresponding
%  to theta_fix will be set to NaN.
%
%  ll - Log likelihood.
%
%  pM - Bayes model posterior, estimated via Laplace
%  integration approximation.
%
%  p - Flag which is 0 whenever Hessian is positive definite.
%  Note! When p>0 then H is not positive definite and the
%  error bars and model posterior can not be trusted.
%
%  normg - The inf-norm of the gradient vector in optimizer 
%  domain. This should be small in combination with p=0 to 
%  indicate the the model is fitted to an optimum.
%
%  H - Hessian matrix, in human domain.
% 
%  g - gradient vector, in open/optimizer domain.
%
%  U - The Cholesky factor of H.
%
%  rcondU - Reciprocal 1-norm condition number of U.

if nargin<4, theta_fix = []; end
shutup = 1;
vcf = 1;
[nll1,g1,H1] = tvacost(theta,theta_fix,tvamodel,tvadata,vcf,shutup);

%V1 = 1/sqrt(theta'*H1*theta);

vcf = 0;
[nll0,g0,H0] = tvacost(theta,theta_fix,tvamodel,tvadata,vcf,shutup);
nll=nll0;
g=g1;
H=H0;
normg=max(abs(g(:)));
[U,p] = chol(H);
rcondU = rcond(U);
if p>0
  H
  warning('Hessian is not positive definite.');
end
M = length(theta);
ll = -nll;
pM = ll + log(2*pi)*(M)/2 - sum(log(diag(U)));
invU = inv(U);
if p==0
  sigma = sqrt(diag(invU*invU'));
else
  sigma = nan(size(H,1),1);
end

[theta_comb,nanfix] = tvathetacombine(theta,theta_fix);
% V0
%[humtheta_adapt,humtheta_fix] = tvathetacombust(tvahuman(theta_comb,tvamodel),nanfix);
%V0 = sqrt(det(inv(H)) / norm(humtheta_adapt,2)); %
%V0 = 1/sqrt(humtheta_adapt'*H0*humtheta_adapt);
% end V0
[xx,mm,yy,humstrip] = tvastriptheta(theta_comb,tvamodel);
humtheta = [tvahuman(xx,mm);humstrip];  % make human

if ~isempty(nanfix)
  tmp = nan(size(humtheta));
  tmp(nanfix) = sigma;
  sigma=tmp;
end

%if p>0
%  V1 = NaN;
%  V0 = NaN;
%end
