function [cc,tt,nn,varargout] = tvaconditions(tvadata)
% TVACONDITIONS
%
%  Synopsis
%  ========
%
%  1. for PR and WR data
%
%  [cc,tt,nn,errors,intrusions,   ...
%   reported,correported,targets, ...
%   distractors,scorepdf,maxtargets] = tvaconditions(tvadata)
%
%  2. for CD data
%
%  [cc,tt,nn,tp,fp,tn,fn,dk] = tvaconditions(tvadata);
%
%  -- Author: Mads Dyrholm --
%     Center for Visual Cognition, University of Copenhagen.
%     2009 - May 2010
% 
%  Purpose
%  ======= 
%
%  Summarize stimulus and subject behavior and 
%  performance per condition.
%
%  Inputs
%  ======
%
%  tvadata - Dataset (see also TVALOADER).

isCD = tvaiscddata(tvadata);
isREP = tvaiswrdata(tvadata) | tvaisprdata(tvadata);

if isCD & isREP
  error('This dataset has both CD and *R tasks, a mix which is not supported.');
end

% cc, tt, and nn
huske_c = [];
for trial=1:length(tvadata)
  if isREP, huske_c = [huske_c,tvadata{trial}.more.condition]; 
  elseif isCD, huske_c = [huske_c,tvadata{trial}.condition]; 
  end
end
cc = sort(unique(huske_c));
tt = zeros(size(cc));
nn = zeros(size(cc));
for trial=1:length(tvadata)
  if isREP, cidx = find(tvadata{trial}.more.condition==cc);
  elseif isCD, cidx = find(tvadata{trial}.condition==cc);
  end
  tt(cidx) = tt(cidx) + tvadata{trial}.t;
  nn(cidx) = nn(cidx) + 1;
end
tt = tt./nn;

% task specific outputs
if isREP>0
  % WR or PR
  errors = zeros(size(cc));
  intrusions = zeros(size(cc));
  reported = zeros(size(cc));
  correported = zeros(size(cc));
  targets = zeros(size(cc));
  distractors = zeros(size(cc));
  maxtargets = zeros(size(cc));
  for trial=1:length(tvadata)
    cidx = find(tvadata{trial}.more.condition==cc);
    tmp1 = tvadata{trial}.more.errors_extra;
    tmp2 = tvadata{trial}.more.errors_intrusion;
    tmp3 = tvadata{trial}.more.report_len_unique;
    tmp4 = length(tvadata{trial}.response);
    errors(cidx) = errors(cidx)+tmp1;
    intrusions(cidx) = intrusions(cidx)+tmp2;
    reported(cidx) = reported(cidx)+tmp3; 
    correported(cidx) = correported(cidx)+tmp4;
    targets(cidx) = targets(cidx)+length(tvadata{trial}.targets);
    distractors(cidx) = distractors(cidx)+length(tvadata{trial}.distractors);
    if length(tvadata{trial}.targets)>maxtargets(cidx)
      maxtargets(cidx)=length(tvadata{trial}.targets);
    end
  end
  scorepdf = zeros(length(cc),max(maxtargets)+1);
  for trial=1:length(tvadata)
    cidx = find(tvadata{trial}.more.condition==cc);
    scorepdf(cidx,length(tvadata{trial}.response)+1) = scorepdf(cidx,length(tvadata{trial}.response)+1) + 1;
  end
  varargout = {errors,intrusions,reported,correported,targets,distractors,scorepdf,maxtargets};
elseif isCD>0
  % Change Detection
  tp = zeros(size(cc));
  fp = zeros(size(cc));
  tn = zeros(size(cc));
  fn = zeros(size(cc));
  dk = zeros(size(cc));
  targets = zeros(size(cc));
  distractors = zeros(size(cc));
  for trial=1:length(tvadata)
    cidx = find(tvadata{trial}.condition==cc);
    tp(cidx) = tp(cidx) + (tvadata{trial}.change==1 & tvadata{trial}.response==1);
    fp(cidx) = fp(cidx) + (tvadata{trial}.change==0 & tvadata{trial}.response==1);
    tn(cidx) = tn(cidx) + (tvadata{trial}.change==0 & tvadata{trial}.response==0);
    fn(cidx) = fn(cidx) + (tvadata{trial}.change==1 & tvadata{trial}.response==0);
    dk(cidx) = dk(cidx) + (tvadata{trial}.response==2);
    targets(cidx) = targets(cidx)+length(tvadata{trial}.targets);
    distractors(cidx) = distractors(cidx)+length(tvadata{trial}.distractors);
  end
  varargout = {tp,fp,tn,fn,dk,targets,distractors};
end
